using System;
using System.IO;
using DynaPDF;

namespace examples
{
	class Class1
	{
      /*
         Note that the dynapdf.dll must be copied into the output directory or into a
         Windwos search path (e.g. %WINDOWS%/System32) before the application can be executed!
      */

      // Error callback function.
      // If the function name should not appear at the beginning of the error message then set
      // the flag emNoFuncNames (pdf.SetErrorMode(DynaPDF.TErrMode.emNoFuncNames);).
      static int PDFError(IntPtr Data, int ErrCode, IntPtr ErrMessage, int ErrType)
      {
         Console.Write("{0}\n", System.Runtime.InteropServices.Marshal.PtrToStringAnsi(ErrMessage));
         return 0; // We try to continue if an error occurrs. Any other return value breaks processing.
      }

		[STAThread]
		static void Main(string[] args)
		{
         try
         {
            int f, orientation;
            CPDF pdf = new CPDF();
            // Error messages and warnings are passed to the callback function.
            pdf.SetOnErrorProc(IntPtr.Zero, new DynaPDF.TErrorProc(PDFError));
            pdf.CreateNewPDF(null); // We open the output file later if no error occurrs.

            // Import anything and don't convert pages to templates
            pdf.SetImportFlags(TImportFlags.ifImportAll | TImportFlags.ifImportAsPage);
            if (pdf.OpenImportFile("../../../test_files/rotated_270.pdf", TPwdType.ptOpen, null) < 0)
            {
               pdf = null;
               return;
            }
            pdf.ImportPDFFile(1, 1.0, 1.0);
            pdf.CloseImportFile();

            pdf.SetPageCoords(TPageCoord.pcTopDown);

            // This property moves the coordinate origin into the visible area (default value == false).
            pdf.SetUseVisibleCoords(true);

            // Open page 1 for editing
            pdf.EditPage(1);
               // Check whether the page is rotated.
               if ((orientation = pdf.GetOrientation()) != 0)
               {
                  // SetOrientationEx() rotates the coordinate system into the opposite direction of the page orientation.
                  // There is no guarantee that the contents in a page is rotated in the same way. If the result is wrong
                  // then don"t call this function.
                  pdf.SetOrientationEx(orientation);
               }
               f = pdf.SetFont("Helvetica", TFStyle.fsRegular, 12.0, false, TCodepage.cp1252);
               // We use this font also as list font. In this example we use a bullet as list symbol (character index 144 of the code page 1252).
               //
               // Now it becomes complicated: Since C# uses Unicode as string format, WriteFTextEx() passes an Unicode string to DynaPDF but
               // the Unicode index of the bullet character is hex 2022 and not 144! If we would add the decimal number 8226 to the string, then
               // this number would be interpreted as the number of a numbered list.
               //
               // We have two options: Either we use WriteFTextExA() to output the text and add the decimal number 144 to the string, or we add the
               // Unicode character u2020 to the text and use WriteFTextEx() or WriteFTextExW(). We use the latter variant in this example.
               pdf.SetListFont(f);
               pdf.WriteFTextEx(50.0, 200.0, pdf.GetPageWidth() - 100.0, -1.0, TTextAlign.taJustify,
                  "It is not difficult to edit an imported page but two things must be considered:\n\n\\LI[20,\u2022]\\LD[16]The page's "+
                  "orientation.\\EL#\\LI[20,\u2022]\\LD[12]The coordinate origin. The coordinate origin can be taken from the crop box if present, or from the media box (Left and Bottom).\\EL#\n\\LD[12]"+
                  "Although it is possible to correct the coordinate origin manually, it is much easier to set the property SetUseVisibleCoords() to true. DynaPDF moves the zero point then automatically "+
                  "into the visible area of the page.\n\n"+
                  "The functions GetPageWidth() and GetPageHeight() return then also the logical width or height of the page depending on the orientation and whether a crop box is present.\n\n"+
                  "The handling of rotated pages is a bit more complicated since the orientation is just a property. That means there is no guarantee that the contents is rotated "+
                  "into the opposite direction like the contents in this page. Whether this is the case depends on the creator of the PDF file.\n\n"+
                  "However, by default it is probably best to assume that the contents is rotated. SetOrientationEx() rotates the coordinate system so that we can work with the page as if it was "+
                  "not rotated. If this produces a wrong result then don't call SetOrientationEx().\n\n"+
                  "Now you ask probably yourself whether it is possible to identify the orientation of the contents in a page. The answer is maybe. It is possible to parse a page with ParseContent() "+
                  "and to inspect the transformation matrices but this can produce wrong results especially if a page contains not much contents.");
            pdf.EndPage();

            // No fatal error occurred?
            if (pdf.HaveOpenDoc())
            {
               // We write the output file into the current directory.
               String filePath = Directory.GetCurrentDirectory() + "\\out.pdf";
               // OK, now we can open the output file.
               if (!pdf.OpenOutputFile(filePath)) return;
               if (pdf.CloseFile())
               {
                  Console.Write("PDF file \"{0}\" successfully created!\n", filePath);
                  System.Diagnostics.Process p = new System.Diagnostics.Process();
                  p.StartInfo.FileName = filePath;
                  p.Start();
               }
            }
            pdf = null;
         }catch(Exception e)
         {
            Console.Write(e.Message);
            Console.Read();
         }
		}
	}
}
